#region References

using System;
using System.Data;
using System.Data.SqlClient;

using gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using SProc = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ICTable = gov.va.med.vbecs.Common.VbecsTables.InterfaceControl;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>3/26/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>Data access class providing read/write access to InterfaceControl table containing control parameters for HL7 interfaces.</summary>

	#endregion

	/// <summary>
	/// InterfaceControl
	/// </summary>
	public class InterfaceControl
	{
		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7008"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid instance of InterfaceControl</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7009"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blank constructor for InterfaceControl
		/// </summary>
		private InterfaceControl() {}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4498"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>DataTable with non-zero row count.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4499"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all HL7 interfaces.
		/// </summary>
		/// <returns><see cref="DataTable"/> containing control parameters for all HL7 interfaces, one row per each interface.</returns>
		public static DataTable GetAllHL7Interfaces()
		{
			return Common.StoredProcedure.GetDataTable( SProc.HL7GetInterfaceControlParameters.StoredProcName );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4500"> 
		///		<ExpectedInput>Valid interface name.</ExpectedInput>
		///		<ExpectedOutput>Single DataRow with data for specified interface.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4501"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4506"> 
		///		<ExpectedInput>Non-existent interface name.</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves interface control parameters for a given interface. 
		/// </summary>
		/// <param name="interfaceName">Interface name.</param>
		/// <returns><see cref="DataRow"/> containing control parameters for a single HL7 interface.</returns>
		public static DataRow GetInterfaceControlParameters( string interfaceName )
		{
			if( interfaceName == null )
				throw( new ArgumentNullException( "interfaceName" ) );

			SqlParameter _prm = new SqlParameter( SProc.HL7GetInterfaceControlParameters.InterfaceName, SqlDbType.VarChar );
			_prm.Value = interfaceName;

			return Common.StoredProcedure.GetSingleDataRow( SProc.HL7GetInterfaceControlParameters.StoredProcName, _prm );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4509"> 
		///		<ExpectedInput>DataRow with updated field InterfaceIPAddress.</ExpectedInput>
		///		<ExpectedOutput>InterfaceIPAddress is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4510"> 
		///		<ExpectedInput>DataRow with updated field InterfacePortNumber.</ExpectedInput>
		///		<ExpectedOutput>InterfacePortNumber is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4511"> 
		///		<ExpectedInput>DataRow with updated field InterfaceFacilityId.</ExpectedInput>
		///		<ExpectedOutput>InterfaceFacilityId is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4512"> 
		///		<ExpectedInput>DataRow with updated field VbecsIPAddress.</ExpectedInput>
		///		<ExpectedOutput>VbecsIPAddress is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4513"> 
		///		<ExpectedInput>DataRow with updated field VbecsPortNumber.</ExpectedInput>
		///		<ExpectedOutput>VbecsPortNumber is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4514"> 
		///		<ExpectedInput>DataRow with updated field VbecsFacilityID.</ExpectedInput>
		///		<ExpectedOutput>VbecsFacilityID is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4515"> 
		///		<ExpectedInput>DataRow with updated field AckTimeout.</ExpectedInput>
		///		<ExpectedOutput>AckTimeout is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4516"> 
		///		<ExpectedInput>DataRow with updated field ReTransmitAttempts.</ExpectedInput>
		///		<ExpectedOutput>ReTransmitAttempts is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4517"> 
		///		<ExpectedInput>DataRow with updated field PurgeSuccessfulMessage.</ExpectedInput>
		///		<ExpectedOutput>PurgeSuccessfulMessage is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4518"> 
		///		<ExpectedInput>DataRow with updated field PurgeUnSuccessfulMessage.</ExpectedInput>
		///		<ExpectedOutput>PurgeUnSuccessfulMessage is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4519"> 
		///		<ExpectedInput>DataRow with updated field InterfaceAdministratorName.</ExpectedInput>
		///		<ExpectedOutput>InterfaceAdministratorName is updated in DB.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4505"> 
		///		<ExpectedInput>Null instead of DataRow containing interface control parameters to save. </ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates interface control parameters for a single interface from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> containing interface control data to save to database.</param>
		public static void UpdateInterface( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			SqlParameter[] _prms = 
				{
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfaceControlId, dtRow[ ICTable.InterfaceControlId ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfaceIPAddress, dtRow[ ICTable.InterfaceIPAddress ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfaceDomainName, dtRow[ ICTable.InterfaceDomainName ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfacePortNumber, dtRow[ ICTable.InterfacePortNumber ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfaceFacilityId, dtRow[ ICTable.InterfaceFacilityId ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.VbecsIPAddress, dtRow[ ICTable.VbecsIPAddress ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.VbecsPortNumber, dtRow[ ICTable.VbecsPortNumber ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.VbecsFacilityID, dtRow[ ICTable.VbecsFacilityId ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.AckTimeout, dtRow[ ICTable.AckTimeout ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.ReTransmitAttempts, dtRow[ ICTable.ReTransmitAttempts ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.PurgeSuccessfulMessage, dtRow[ ICTable.PurgeSuccessfulMessage ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.PurgeUnSuccessfulMessage, dtRow[ ICTable.PurgeUnSuccessfulMessage ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.InterfaceAdministratorName, dtRow[ ICTable.InterfaceAdministratorName ] ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlParameters.LogEventsIndicator, dtRow[ ICTable.LogEventsIndicator ] ),
					Common.StoredProcedure.MakeLastUpdateUserSqlParam()
				};

			new Common.StoredProcedure().TransactionalGetValue( SProc.HL7UpdateInterfaceControlParameters.StoredProcName, _prms );
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/5/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8835"> 
		///		<ExpectedInput>Valid InterfaceControlId</ExpectedInput>
		///		<ExpectedOutput>Updated value of interface status</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8836"> 
		///		<ExpectedInput>Invalid InterfaceControlId</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates interface control status for a single interface
		/// CR 2940
		/// </summary>
		public static void UpdateInterfaceStatus( int interfaceControlId, bool statusEnabled )
		{
			SqlParameter[] _prms = 
				{
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlStatus.InterfaceControlId, interfaceControlId ),
					Common.StoredProcedure.MakeParameter( SProc.HL7UpdateInterfaceControlStatus.InterfaceActiveIndicator, statusEnabled ),
					Common.StoredProcedure.MakeLastUpdateUserSqlParam()
				};

			new Common.StoredProcedure().TransactionalGetValue( SProc.HL7UpdateInterfaceControlStatus.StoredProcName, _prms );
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8522"> 
		///		<ExpectedInput>InterfaceName enum</ExpectedInput>
		///		<ExpectedOutput>InterfaceActiveStatus</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8523"> 
		///		<ExpectedInput>Na</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2940
		/// </summary>
		public static InterfaceActiveStatus GetInterfaceActiveIndicator( Common.InterfaceName interfaceName )
		{
			SqlParameter[] prms =
			{
				new SqlParameter( SProc.GetInterfaceActiveIndicator.interfacename, System.Data.SqlDbType.VarChar ),
			};
			//
			prms[0].Value = interfaceName.ToString();
			//
			object interfaceActiveStatus = StoredProcedure.GetSingleDataRow( SProc.GetInterfaceActiveIndicator.StoredProcName, prms )[ICTable.InterfaceActiveIndicator];
			//
			if ( interfaceActiveStatus == System.DBNull.Value )
			{
				return InterfaceActiveStatus.NotSet;
			}
			else
			{
				if ( Convert.ToBoolean( interfaceActiveStatus ) )
				{
					return InterfaceActiveStatus.Active;
				}
				else
				{
					return InterfaceActiveStatus.Inactive;
				}
			}
		}
	}
}